<#
.SYNOPSIS
Enable automatic version trimming on document libraries and perform a one-time prune
(keeping only the most recent N versions) across:
- All SharePoint sites created on or before a cutoff date, OR
- A single specified site (using -SingleSiteUrl).

.DESCRIPTION
- Supports Delegated (interactive) and AppOnly (certificate-based) authentication with PnP.PowerShell.
- Delegated uses: Connect-PnPOnline -Interactive -ClientId.
- AppOnly uses: certificate thumbprint or PFX file (preferred over client secret).
- Grants Site Collection Admin on target sites by default to avoid unauthorized errors:
  - Delegated: ensures the current user is owner via tenant admin cmdlet.
  - AppOnly: ensures a specified human UPN via -EnsureOwnerUpn (recommended).
- Elevation uses Set-PnPTenantSite -Owners from the admin connection for efficiency.
- Enables Enhanced Version Controls (Auto Expiration Version Trim) on each document library:
  Set-PnPList -EnableAutoExpirationVersionTrim $true
- Performs a one-time cleanup to keep only the latest N versions per file using:
  Get-PnPFileVersion / Remove-PnPFileVersion.
- Logs to C:\Temp\ErrorLog_<MM-dd-yyyy>.csv using your preferred Write-Log function.
- Supports detailed (per-library & per-file) logging via -DetailedLog.
- Generates an aggregated summary CSV when -GenerateSummary is specified.

.PARAMETER TenantName
Tenant short name (e.g., contoso).

.PARAMETER ClientId
Azure AD App (public or confidential client) ID.

.PARAMETER AuthMode
Authentication mode: Delegated (default) or AppOnly.

.PARAMETER CertificateThumbprint
Thumbprint for certificate in CurrentUser\My or LocalMachine\My (AppOnly).

.PARAMETER CertificatePath
Path to PFX certificate file (AppOnly).

.PARAMETER CertificatePassword
SecureString password for the PFX (AppOnly).

.PARAMETER EnsureOwnerUpn
In AppOnly mode, human UPN to ensure as Site Collection Admin on target sites.

.PARAMETER CutoffDate
Process all sites created on or before this date (ignored if -SingleSiteUrl is provided).

.PARAMETER KeepLatestVersions
Number of most recent versions to keep during one-time prune (default: 10).

.PARAMETER DryRun
Simulate actions without making any changes (still elevates admin to ensure access, by design).

.PARAMETER SingleSiteUrl
Optional. If provided, the script runs only against this site and skips tenant-wide enumeration.

.PARAMETER DetailedLog
Optional. If provided, logs detailed per-library and per-file actions.

.PARAMETER GenerateSummary
Optional. If provided, writes an aggregated CSV report:
C:\Temp\VersionTrimSummary_<yyyyMMdd_HHmmss>.csv

.PARAMETER CleanupAdminAccess
Optional. If provided, removes Site Collection Admin for the identity elevated by this run.

.EXAMPLES
# Example 1: DELEGATED - All sites created on/before 8/26/2025 (Dry Run, summary)
.\Pnp-VersionHistoryPurge_SetAutomaticTrim.ps1 `
  -TenantName "contoso" `
  -ClientId "00000000-0000-0000-0000-000000000000" `
  -CutoffDate '2025-08-26' `
  -KeepLatestVersions 10 `
  -DryRun `
  -GenerateSummary

# Example 2: DELEGATED - Single site, detailed logging
.\Pnp-VersionHistoryPurge_SetAutomaticTrim.ps1 `
  -TenantName "contoso" `
  -ClientId "00000000-0000-0000-0000-000000000000" `
  -SingleSiteUrl "https://contoso.sharepoint.com/sites/Finance" `
  -KeepLatestVersions 10 `
  -DetailedLog `
  -GenerateSummary

# Example 3: APPONLY - Thumbprint-based certificate
.\Pnp-VersionHistoryPurge_SetAutomaticTrim.ps1 `
  -TenantName "contoso" `
  -ClientId "00000000-0000-0000-0000-000000000000" `
  -AuthMode AppOnly `
  -CertificateThumbprint "ABCD1234EF567890ABCD1234EF567890ABCD1234" `
  -SingleSiteUrl "https://contoso.sharepoint.com/sites/Finance" `
  -KeepLatestVersions 10 `
  -DetailedLog `
  -GenerateSummary

# Example 4: APPONLY - PFX file
$certPwd = Read-Host -AsSecureString "Enter PFX password"
.\Pnp-VersionHistoryPurge_SetAutomaticTrim.ps1 `
  -TenantName "contoso" `
  -ClientId "00000000-0000-0000-0000-000000000000" `
  -AuthMode AppOnly `
  -CertificatePath "C:\Certs\ContosoAppCert.pfx" `
  -CertificatePassword $certPwd `
  -CutoffDate '2025-08-26' `
  -KeepLatestVersions 10 `
  -GenerateSummary

                                         *** Disclamer ***
 
The sample scripts are not supported under any AdaptivEdge standard support program or service. The sample
scripts are provided AS IS without warranty of any kind. AdptivEdge further disclaims all implied warranties
including, without limitation, any implied warranties of merchantability or of fitness for a particular
purpose. The entire risk arising out of the use or performance of the sample scripts and documentation
remains with you. In no event shall AdaptivEdge, its authors, or anyone else involved in the creation,
production, or delivery of the scripts be liable for any damages whatsoever (including, without limitation,
damages for loss of business profits, business interruption, loss of business information, or other
pecuniary loss) arising out of the use of or inability to use the sample scripts or documentation, even
if AdaptivEdge has advised of the possibility of such damages.
 
Original script by: AdaptivEdge LLC

#>

[CmdletBinding()]
param(
    [Parameter(Mandatory=$true)][string]$TenantName,
    [Parameter(Mandatory=$true)][string]$ClientId,

    [ValidateSet('Delegated','AppOnly')]
    [string]$AuthMode = 'Delegated',

    # App-only certificate options
    [string]$CertificateThumbprint,
    [string]$CertificatePath,
    [SecureString]$CertificatePassword,

    # Human owner to ensure in AppOnly mode (optional if app already has full access)
    [string]$EnsureOwnerUpn,

    [datetime]$CutoffDate = [datetime]'2025-08-26',
    [int]$KeepLatestVersions = 500,
    [switch]$DryRun,
    [string]$SingleSiteUrl,
    [switch]$DetailedLog,
    [switch]$GenerateSummary,
    [switch]$CleanupAdminAccess
)

$ErrorActionPreference = 'Stop'
$ProgressPreference    = 'SilentlyContinue'

# Ensure log folder exists
if (-not (Test-Path 'C:\Temp')) { New-Item -Path 'C:\Temp' -ItemType Directory -Force | Out-Null }

# Logging to C:\Temp\ErrorLog_<MM-dd-yyyy>.csv
$GlobalErrorLogPath = "C:\Temp\ErrorLog_{0}.csv" -f (Get-Date -Format 'MM-dd-yyyy')

Function Write-Log {
    param (
        [string]$SiteUrl,
        [string]$Action,
        [string]$Status,
        [string]$ErrorMessage = ""
    )

    # Ensure log folder exists
    if (-not (Test-Path 'C:\Temp')) {
        New-Item -Path 'C:\Temp' -ItemType Directory -Force | Out-Null
    }

    # Initialize CSV with headers if it doesn't exist
    if (-not (Test-Path $GlobalErrorLogPath)) {
        $null = [pscustomobject]@{
            Timestamp = "Timestamp"
            SiteUrl   = "SiteUrl"
            Action    = "Action"
            Status    = "Status"
            Details   = "Details"
        } | Export-Csv -Path $GlobalErrorLogPath -NoTypeInformation -Encoding UTF8
    }

    # Prepare log entry
    $timestamp  = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $cleanError = if ($null -ne $ErrorMessage) { ($ErrorMessage -replace '[\r\n]+',' ') } else { "" }

    $logEntry = [pscustomobject]@{
        Timestamp = $timestamp
        SiteUrl   = $SiteUrl
        Action    = $Action
        Status    = $Status
        Details   = $cleanError
    }

    # Append to CSV 
    $logEntry | Export-Csv -Path $GlobalErrorLogPath -NoTypeInformation -Encoding UTF8 -Append

    # Console output
    if ($Status -eq "Success") {
        Write-Host "$timestamp [$Action] - $SiteUrl Success" -ForegroundColor Green
    } else {
        Write-Host "$timestamp [$Action] - $SiteUrl Error - $cleanError" -ForegroundColor Red
    }
}


# Summary output
if ($GenerateSummary) {
    $script:SummaryRows = New-Object System.Collections.Generic.List[object]
    $script:SummaryPath = "C:\Temp\VersionTrimSummary_{0}.csv" -f (Get-Date -Format 'yyyyMMdd_HHmmss')
}

# Tracking for cleanup
$script:SitesWeElevated     = New-Object System.Collections.Generic.List[string]
$script:EnsuredOwnersBySite = @{}

# Derived values
$TenantDomain = "$TenantName.onmicrosoft.com"
$AdminUrl     = "https://$TenantName-admin.sharepoint.com"

# ===== Authentication (no client secret paths) =====
function Connect-PnPAdmin {
    try {
        # Safely disconnect any existing session
        try {
            $null = Get-PnPConnection
            Disconnect-PnPOnline -ErrorAction SilentlyContinue | Out-Null
        } catch { }

        # Connect based on auth mode
        if ($AuthMode -eq 'Delegated') {
            Connect-PnPOnline -Url $AdminUrl -ClientId $ClientId -Interactive | Out-Null
        } else {
            if ($CertificateThumbprint) {
                Connect-PnPOnline -Url $AdminUrl -ClientId $ClientId -Tenant $TenantDomain -Thumbprint $CertificateThumbprint | Out-Null
            } elseif ($CertificatePath) {
                if ($CertificatePassword) {
                    Connect-PnPOnline -Url $AdminUrl -ClientId $ClientId -Tenant $TenantDomain -CertificatePath $CertificatePath -CertificatePassword $CertificatePassword | Out-Null
                } else {
                    Connect-PnPOnline -Url $AdminUrl -ClientId $ClientId -Tenant $TenantDomain -CertificatePath $CertificatePath | Out-Null
                }
            } else {
                throw "AppOnly selected but no certificate provided. Supply -CertificateThumbprint or -CertificatePath (+ -CertificatePassword)."
            }
        }

        Write-Log -SiteUrl "$AdminUrl" -Action "Connect-PnPAdmin ($AuthMode)" -Status "Success"
    } catch {
        Write-Log -SiteUrl "$AdminUrl" -Action "Connect-PnPAdmin ($AuthMode)" -Status "Error" -ErrorMessage $_.Exception.Message
        throw
    }
}

function Connect-PnPSite {
    param([string]$SiteUrl)
    try {
        if ($AuthMode -eq 'Delegated') {
            Connect-PnPOnline -Url $SiteUrl -ClientId $ClientId -Interactive | Out-Null
        } else {
            if ($CertificateThumbprint) {
                Connect-PnPOnline -Url $SiteUrl -ClientId $ClientId -Tenant $TenantDomain -Thumbprint $CertificateThumbprint | Out-Null
            } elseif ($CertificatePath) {
                if ($CertificatePassword) {
                    Connect-PnPOnline -Url $SiteUrl -ClientId $ClientId -Tenant $TenantDomain -CertificatePath $CertificatePath -CertificatePassword $CertificatePassword | Out-Null
                } else {
                    Connect-PnPOnline -Url $SiteUrl -ClientId $ClientId -Tenant $TenantDomain -CertificatePath $CertificatePath | Out-Null
                }
            } else {
                throw "AppOnly selected but no certificate provided. Supply -CertificateThumbprint or -CertificatePath (+ -CertificatePassword)."
            }
        }
        Write-Log -SiteUrl "$SiteUrl" -Action "Connect-PnPSite ($AuthMode)" -Status "Success"
    } catch {
        Write-Log -SiteUrl "$SiteUrl" -Action "Connect-PnPSite ($AuthMode)" -Status "Error" -ErrorMessage $_.Exception.Message
        throw
    }
}

# ===== Target Sites =====
function Get-TargetSites {
    try {
        $sites = Get-PnPTenantSite -IncludeOneDriveSites:$false | Where-Object { $_.Template -ne "REDIRECTSITE#0" }
        $filtered = $sites | Where-Object {
            $created = $_.CreationTime; if (-not $created) { $created = $_.Created }
            $isSystem = $_.Template -in @('APPCATALOG#0','SPSMSITEHOST#0')
            ($created -le $CutoffDate) -and (-not $isSystem)
        }
        Write-Log -SiteUrl "$AdminUrl" -Action "Get-TargetSites" -Status "Success" -ErrorMessage ("Count=" + ($filtered | Measure-Object).Count)
        return $filtered
    } catch {
        Write-Log -SiteUrl "$AdminUrl" -Action "Get-TargetSites" -Status "Error" -ErrorMessage $_.Exception.Message
        throw
    }
}

# ===== Elevation Helpers =====
function Resolve-OwnerToEnsure {
    if ($AuthMode -eq 'Delegated') {
        try {
            $meLogin = (Get-PnPWeb -Includes CurrentUser).CurrentUser.LoginName
            if (-not $meLogin) { throw "Unable to resolve current user login on admin site." }
            Write-Log -SiteUrl "$AdminUrl" -Action "CurrentUser" -Status "Success" -ErrorMessage $meLogin
            return $meLogin
        } catch {
            Write-Log -SiteUrl "$AdminUrl" -Action "CurrentUser" -Status "Error" -ErrorMessage $_.Exception.Message
            return $null
        }
    } else {
        if ($EnsureOwnerUpn) {
            Write-Log -SiteUrl "$AdminUrl" -Action "EnsureOwnerUpn" -Status "Success" -ErrorMessage $EnsureOwnerUpn
            return $EnsureOwnerUpn
        } else {
            # With Sites.FullControl.All, elevation can be skipped
            Write-Log -SiteUrl "$AdminUrl" -Action "EnsureOwnerUpn" -Status "Error" -ErrorMessage "AppOnly: no -EnsureOwnerUpn provided. Skipping elevation."
            return $null
        }
    }
}

function Ensure-AdminAccessForSites {
    param([string[]]$SiteUrls)

    $ownerToEnsure = Resolve-OwnerToEnsure
    foreach ($url in $SiteUrls) {
        try {
            if ($null -eq $ownerToEnsure) {
                Write-Log -SiteUrl "$url" -Action "AddSiteCollectionAdmin" -Status "Success" -ErrorMessage "Skipped (no owner to ensure in $AuthMode)"
                continue
            }

            # Ensure via admin context
            Set-PnPTenantSite -Identity $url -Owners $ownerToEnsure
            [void]$script:SitesWeElevated.Add($url)
            $script:EnsuredOwnersBySite[$url] = $ownerToEnsure

            Write-Log -SiteUrl "$url" -Action "AddSiteCollectionAdmin" -Status "Success" -ErrorMessage "Ensured '$ownerToEnsure' via Set-PnPTenantSite -Owners"
        } catch {
            Write-Log -SiteUrl "$url" -Action "AddSiteCollectionAdmin" -Status "Error" -ErrorMessage $_.Exception.Message
        }
    }
}

function Cleanup-AdminAccessForSites {
    param([string[]]$SiteUrls)
    if (-not $CleanupAdminAccess) { return }

    foreach ($url in $SiteUrls) {
        if ($script:SitesWeElevated -notcontains $url) {
            Write-Log -SiteUrl "$url" -Action "RemoveSiteCollectionAdmin" -Status "Success" -ErrorMessage "Skipped removal (not elevated by this run)"
            continue
        }

        $ownerToRemove = $null
        if ($script:EnsuredOwnersBySite.ContainsKey($url)) {
            $ownerToRemove = $script:EnsuredOwnersBySite[$url]
        } elseif ($AuthMode -eq 'Delegated') {
            try { $ownerToRemove = (Get-PnPWeb -Includes CurrentUser).CurrentUser.LoginName } catch {}
        } else {
            $ownerToRemove = $EnsureOwnerUpn
        }

        if (-not $ownerToRemove) {
            Write-Log -SiteUrl "$url" -Action "RemoveSiteCollectionAdmin" -Status "Error" -ErrorMessage "Unknown owner to remove in $AuthMode"
            continue
        }

        try {
            # Switch to site context to remove
            Connect-PnPSite -SiteUrl $url
            Remove-PnPSiteCollectionAdmin -Owners $ownerToRemove
            Write-Log -SiteUrl "$url" -Action "RemoveSiteCollectionAdmin" -Status "Success" -ErrorMessage "Removed $ownerToRemove from Site Collection Admins"
            # Switch back to admin for safety if more tenant ops follow
            Connect-PnPAdmin
        } catch {
            Write-Log -SiteUrl "$url" -Action "RemoveSiteCollectionAdmin" -Status "Error" -ErrorMessage $_.Exception.Message
        }
    }
}

# ===== Library & Versioning =====
function Set-LibraryAutoExpiration {
    param([Microsoft.SharePoint.Client.List]$List, [string]$SiteUrl)
    try {
        if ($DryRun) {
            if ($DetailedLog) {
                Write-Log -SiteUrl "$SiteUrl" -Action ("Set AutoExpiration -> " + $List.Title) -Status "Success" -ErrorMessage "[DryRun] Set-PnPList -EnableAutoExpirationVersionTrim $true"
            }
        } else {
            Set-PnPList -Identity $List -EnableAutoExpirationVersionTrim $true
            if ($DetailedLog) {
                Write-Log -SiteUrl "$SiteUrl" -Action ("Set AutoExpiration -> " + $List.Title) -Status "Success"
            }
        }
    } catch {
        Write-Log -SiteUrl "$SiteUrl" -Action ("Set AutoExpiration -> " + $List.Title) -Status "Error" -ErrorMessage $_.Exception.Message
    }
}

function Trim-FileVersionsInList {
    param([Microsoft.SharePoint.Client.List]$List, [string]$SiteUrl, [int]$Keep = 10)

    $filesProcessed  = 0
    $versionsDeleted = 0

    try {
        $items = Get-PnPListItem -List $List -PageSize 2000 -Fields FileLeafRef,FileRef,FSObjType |
                 Where-Object { $_.FileSystemObjectType -eq "File" }

        foreach ($it in $items) {
            $fileUrl = $it.FieldValues.FileRef
            $filesProcessed++

            try {
                $versions = Get-PnPFileVersion -Url $fileUrl
                if (-not $versions) { continue }

                $toDelete      = $versions | Sort-Object -Property Created -Descending | Select-Object -Skip $Keep
                $countToDelete = ($toDelete | Measure-Object).Count

                if ($countToDelete -gt 0) {
                    if ($DryRun) {
                        if ($DetailedLog) {
                            Write-Log -SiteUrl "$SiteUrl" -Action ("DryRun DeleteVersions -> " + $fileUrl) -Status "Success" -ErrorMessage ("Would remove " + $countToDelete + " older versions")
                        }
                    } else {
                        foreach ($v in $toDelete) {
                            Remove-PnPFileVersion -Url $fileUrl -Identity $v.Id -Force
                        }
                        if ($DetailedLog) {
                            Write-Log -SiteUrl "$SiteUrl" -Action ("DeletedVersions -> " + $fileUrl) -Status "Success" -ErrorMessage ("Removed " + $countToDelete + " older versions")
                        }
                    }
                    $versionsDeleted += $countToDelete
                }
            } catch {
                Write-Log -SiteUrl "$SiteUrl" -Action ("TrimError -> " + $fileUrl) -Status "Error" -ErrorMessage $_.Exception.Message
            }
        }
    } catch {
        Write-Log -SiteUrl "$SiteUrl" -Action ("Prune Error -> " + $List.Title) -Status "Error" -ErrorMessage $_.Exception.Message
    }

    return [pscustomobject]@{
        SiteUrl         = $SiteUrl
        LibraryName     = $List.Title
        FilesProcessed  = $filesProcessed
        VersionsDeleted = $versionsDeleted
        Mode            = $(if ($DryRun) { 'DryRun' } else { 'Actual' })
    }
}

function Process-Site {
    param([string]$SiteUrl)

    try {
        Connect-PnPSite -SiteUrl $SiteUrl
    } catch {
        # Connection already logged
        return
    }

    $excluded = @(
        "Form Templates","Preservation Hold Library","Site Assets","Style Library",
        "Site Pages","Images","Site Collection Documents","Site Collection Images"
    )
    try {
        $libs = Get-PnPList -ErrorAction Stop | Where-Object {
            $_.BaseType -eq "DocumentLibrary" -and $_.Hidden -eq $false -and $_.Title -notin $excluded
        }
    } catch {
        Write-Log -SiteUrl "$SiteUrl" -Action "Get-PnPList" -Status "Error" -ErrorMessage "Unauthorized or failed to enumerate libraries. Skipping site."
        return
    }

    Write-Log -SiteUrl "$SiteUrl" -Action "ProcessSite" -Status "Success" -ErrorMessage ("Libraries=" + $libs.Count)

    foreach ($lib in $libs) {
        Set-LibraryAutoExpiration -List $lib -SiteUrl $SiteUrl
        $row = Trim-FileVersionsInList -List $lib -SiteUrl $SiteUrl -Keep $KeepLatestVersions
        if ($GenerateSummary -and $null -ne $row) { [void]$script:SummaryRows.Add($row) }
    }
}

# ===== Main =====
Write-Log -SiteUrl "$AdminUrl" -Action "Start" -Status "Success"

Connect-PnPAdmin

# Determine target sites
[string[]]$targetSiteUrls = @()
if ($SingleSiteUrl) {
    $targetSiteUrls = @($SingleSiteUrl)
    Write-Log -SiteUrl "$SingleSiteUrl" -Action "Mode" -Status "Success" -ErrorMessage "Single site"
} else {
    $sites = Get-TargetSites
    $targetSiteUrls = $sites.Url
}

# Ensure admin access (always, even during DryRun)
Ensure-AdminAccessForSites -SiteUrls $targetSiteUrls

# Process each site
if ($SingleSiteUrl) {
    try { Process-Site -SiteUrl $SingleSiteUrl } catch { Write-Log -SiteUrl "$SingleSiteUrl" -Action "Process-Site" -Status "Error" -ErrorMessage $_.Exception.Message }
} else {
    foreach ($u in $targetSiteUrls) {
        try { Process-Site -SiteUrl $u } catch { Write-Log -SiteUrl "$u" -Action "Process-Site" -Status "Error" -ErrorMessage $_.Exception.Message }
    }
}

# Optional summary report
if ($GenerateSummary) {
    try {
        $script:SummaryRows |
            Sort-Object SiteUrl, LibraryName |
            Export-Csv -Path $script:SummaryPath -NoTypeInformation -Encoding UTF8
        Write-Log -SiteUrl "$AdminUrl" -Action "SummaryReport" -Status "Success" -ErrorMessage $script:SummaryPath
    } catch {
        Write-Log -SiteUrl "$AdminUrl" -Action "SummaryReport" -Status "Error" -ErrorMessage $_.Exception.Message
    }
}

# Optional cleanup (only for sites elevated by this run)
Cleanup-AdminAccessForSites -SiteUrls $targetSiteUrls

